library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity cpu_core0 is
    generic (
        number: integer := 0
    );
    
    Port (
        clk       : in  std_logic;                   -- 1 Hz-Clock
        accu_out  : out std_logic_vector(15 downto 0);
        debug_cmd : out std_logic_vector(15 downto 0); -- One‐hot: LDA, ADD, STA, BA
        led_out   : out std_logic_vector(7 downto 0);  -- neu: Low Byte für LEDs
        inputRegister: in std_logic_vector(8 downto 0);
        outputRegister: out std_logic_vector(7 downto 0)
    );
end cpu_core0;

architecture Behavioral of cpu_core0 is
    
    constant DEBUG_ADDR : integer := 600;
    
    type state_type is (INIT, FETCH, READ_WAIT_FETCH, DECODE, READ_WAIT_FIRST_OPERAND, READ_FIRST_OPERAND, READ_WAIT_SECOND_OPERAND, READ_SECOND_OPERAND, READ_WAIT_THIRD_OPERAND, READ_THIRD_OPERAND, EXECUTE);
    signal state        : state_type := INIT;

    signal pc           : unsigned(9 downto 0) := (others => '0');
    signal ir           : std_logic_vector(15 downto 0) := (others => '0');
    signal accu         : std_logic_vector(15 downto 0) := (others => '0');
    signal stack        : unsigned(9 downto 0) := (others => '0');
    signal carryFlag    : std_logic := '0';

    signal mem_addr     : unsigned(9 downto 0);
    signal mem_din      : std_logic_vector(15 downto 0);
    signal mem_dout     : std_logic_vector(15 downto 0);
    signal mem_we       : std_logic := '0';
    signal stop         : std_logic := '0';

    signal operand      : std_logic_vector(15 downto 0);
    signal init_counter : integer range 0 to 1024 := 0;
    signal reads_left   : integer range 0 to 3 := 0;

    signal debug_reg    : std_logic_vector(15 downto 0) := (others => '0');
    signal led_reg      : std_logic_vector(7 downto 0) := (others => '0');  -- LED-Latch
    
    signal regInputRegister : std_logic_vector(8 downto 0);
    signal regOutputRegister : std_logic_vector(7 downto 0);
    
    -- Number of operands
    function number_operandReads(op : std_logic_vector(15 downto 0)) return integer is
        variable typ : std_logic_vector(3 downto 0) := op(15 downto 12);
        variable adr : std_logic_vector(1 downto 0) := op(11 downto 10);
    begin
        case typ is
            when "1111" | "0110" =>  -- NOP, RET
                return 0;

            when "0100" | "0101" =>  -- Branch, CALL
                return 1;

            when "0001" | "0011" =>  -- LD, MOP
                case adr is
                    when "01" => return 1;  -- X
                    when "10" => return 2;  -- M(X)
                    when "11" => return 3;  -- M(M(X))
                    when others => return 0; -- "00" ungenutzt
                end case;

            when "0010" =>           -- ST
                case adr is
                    when "10" => return 1;  -- STAM  M(X)
                    when "11" => return 2;  -- STAMM M(M(X))
                    -- when "01" => return 1;
                    when others => return 0;
                end case;

            when others =>
                return 0;
        end case;
    end function;

begin

    -- RAM-Instanz
    mem_inst : entity work.mem_ip
    port map(
        clock   => clk,
        address => std_logic_vector(mem_addr),
        data    => mem_din,
        wren    => mem_we,
        q       => mem_dout,
        inputRegister => inputRegister
    );

    -- Ausgänge
    accu_out  <= accu(15 downto 0);
    debug_cmd <= debug_reg;
    led_out   <= led_reg;
    
    process(clk)
    variable r : integer := 0;
    variable next_accu : unsigned(16 downto 0);
    variable writeAccu : boolean := false;
    begin
        if rising_edge(clk) then
            mem_we    <= '0';
            -- debug_reg <= (others => '0');  -- Default: keine LED
            writeAccu := false;
            next_accu := (others => '0');
            
            case state is
                ------------------------------------------------------------
                when INIT =>
                    case init_counter is
-- Programmstart
-- *****************************************

                        when 0 => mem_addr <= to_unsigned(0, 10);  mem_din <= x"4000"; mem_we <= '1';
                        when 1 => mem_addr <= to_unsigned(1, 10);  mem_din <= x"02BC"; mem_we <= '1';   -- 700
                            
-- *************************************************************************************************************
-- SQRT: Parameter
-- *************************************************************************************************************

                        when 4 => mem_addr <= to_unsigned(4, 10);  mem_din <= x"0000"; mem_we <= '1';  -- left
                        when 5 => mem_addr <= to_unsigned(5, 10);  mem_din <= x"00FF"; mem_we <= '1';  -- right
                        when 6 => mem_addr <= to_unsigned(6, 10);  mem_din <= x"0000"; mem_we <= '1';  -- sqrt
                        when 7 => mem_addr <= to_unsigned(7, 10);  mem_din <= x"0000"; mem_we <= '1';  -- cmp
                        when 8 => mem_addr <= to_unsigned(8, 10);  mem_din <= x"0000"; mem_we <= '1';  -- quadrat
                        when 9 => mem_addr <= to_unsigned(9, 10);  mem_din <= x"0000"; mem_we <= '1';  -- middle

-- SQRT
-- *****************************************

-- check if left <= right
                        when 10 => mem_addr <= to_unsigned(10, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 11 => mem_addr <= to_unsigned(11, 10);  mem_din <= x"0004"; mem_we <= '1';  -- left

                        when 12 => mem_addr <= to_unsigned(12, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 13 => mem_addr <= to_unsigned(13, 10);  mem_din <= x"001D"; mem_we <= '1';  -- copy a to add left + reight

                        when 14 => mem_addr <= to_unsigned(14, 10);  mem_din <= x"3540"; mem_we <= '1';  -- NOT
                        when 15 => mem_addr <= to_unsigned(15, 10);  mem_din <= x"0000"; mem_we <= '1';

                        when 16 => mem_addr <= to_unsigned(16, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        when 17 => mem_addr <= to_unsigned(17, 10);  mem_din <= x"0001"; mem_we <= '1';

                        when 18 => mem_addr <= to_unsigned(18, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 19 => mem_addr <= to_unsigned(19, 10);  mem_din <= x"0017"; mem_we <= '1';  -- copy a to add

                        when 20 => mem_addr <= to_unsigned(20, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 21 => mem_addr <= to_unsigned(21, 10);  mem_din <= x"0005"; mem_we <= '1';  -- right

                        when 22 => mem_addr <= to_unsigned(22, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        when 23 => mem_addr <= to_unsigned(23, 10);  mem_din <= x"0000"; mem_we <= '1';

                        when 24 => mem_addr <= to_unsigned(24, 10);  mem_din <= x"4010"; mem_we <= '1';  -- BN
                        when 25 => mem_addr <= to_unsigned(25, 10);  mem_din <= x"0066"; mem_we <= '1';  -- finished

-- Calculate middle

                        when 26 => mem_addr <= to_unsigned(26, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 27 => mem_addr <= to_unsigned(27, 10);  mem_din <= x"0005"; mem_we <= '1';  -- left

                        when 28 => mem_addr <= to_unsigned(28, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        when 29 => mem_addr <= to_unsigned(29, 10);  mem_din <= x"0000"; mem_we <= '1';

                        when 30 => mem_addr <= to_unsigned(30, 10);  mem_din <= x"3580"; mem_we <= '1';  -- SHR
                        when 31 => mem_addr <= to_unsigned(31, 10);  mem_din <= x"0000"; mem_we <= '1';  -- (left+right)>>1

                        when 32 => mem_addr <= to_unsigned(32, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 33 => mem_addr <= to_unsigned(33, 10);  mem_din <= x"0009"; mem_we <= '1';  -- store middle

-- Calculate quadrat

                        when 34 => mem_addr <= to_unsigned(34, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 35 => mem_addr <= to_unsigned(35, 10);  mem_din <= x"00C8"; mem_we <= '1';  -- store in multiply parameter 1

                        when 36 => mem_addr <= to_unsigned(36, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 37 => mem_addr <= to_unsigned(37, 10);  mem_din <= x"00C9"; mem_we <= '1';  -- store in multiply parameter 2

                        when 38 => mem_addr <= to_unsigned(38, 10);  mem_din <= x"1400"; mem_we <= '1';  -- LDA
                        when 39 => mem_addr <= to_unsigned(39, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 0 into Accu

                        when 40 => mem_addr <= to_unsigned(40, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 41 => mem_addr <= to_unsigned(41, 10);  mem_din <= x"00CA"; mem_we <= '1';  -- store 0 in Multiply result

                        when 42 => mem_addr <= to_unsigned(42, 10);  mem_din <= x"5000"; mem_we <= '1';  -- CALL
                        when 43 => mem_addr <= to_unsigned(43, 10);  mem_din <= x"00CB"; mem_we <= '1';  -- MUL

                        when 44 => mem_addr <= to_unsigned(44, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 45 => mem_addr <= to_unsigned(45, 10);  mem_din <= x"0008"; mem_we <= '1';  -- store quadrat

-- -------------- Now check quadrat > n , quadrat == n or quadrat <= n -------


                        when 46 => mem_addr <= to_unsigned(46, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM n
                        when 47 => mem_addr <= to_unsigned(47, 10);  mem_din <= x"00F0"; mem_we <= '1';  -- b bei daten[101]
                        when 48 => mem_addr <= to_unsigned(48, 10);  mem_din <= x"3540"; mem_we <= '1';  -- NOT A       ; A = ~b
                        when 49 => mem_addr <= to_unsigned(49, 10);  mem_din <= x"0000"; mem_we <= '1';
                        when 50 => mem_addr <= to_unsigned(50, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM 9      ; daten[9] = ~b
                        when 51 => mem_addr <= to_unsigned(51, 10);  mem_din <= x"0039"; mem_we <= '1';
                        when 52 => mem_addr <= to_unsigned(52, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM a      ; A = a
                        when 53 => mem_addr <= to_unsigned(53, 10);  mem_din <= x"0008"; mem_we <= '1';  -- a bei daten[100]
                        when 54 => mem_addr <= to_unsigned(54, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD 1       ; A = a – b
                        when 55 => mem_addr <= to_unsigned(55, 10);  mem_din <= x"0001"; mem_we <= '1';
                        when 56 => mem_addr <= to_unsigned(56, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD [n]     ; A = a + (~b)
                        when 57 => mem_addr <= to_unsigned(57, 10);  mem_din <= x"0001"; mem_we <= '1';  -- wird von STAM(9) mit ~b ersetzt

                        when 58 => mem_addr <= to_unsigned(58, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ → 300    ; if A==0   ⇒ a==b
                        when 59 => mem_addr <= to_unsigned(59, 10);  mem_din <= x"004C"; mem_we <= '1';
                        when 60 => mem_addr <= to_unsigned(60, 10);  mem_din <= x"4008"; mem_we <= '1';  -- BC → 100    ; if Carry ⇒ a>b
                        when 61 => mem_addr <= to_unsigned(61, 10);  mem_din <= x"0052"; mem_we <= '1';
                        when 62 => mem_addr <= to_unsigned(62, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA → 200    ; else      a<b
                        when 63 => mem_addr <= to_unsigned(63, 10);  mem_din <= x"005C"; mem_we <= '1';


                        when 70 => mem_addr <= to_unsigned(70, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ
                        when 71 => mem_addr <= to_unsigned(71, 10);  mem_din <= x"004C"; mem_we <= '1';  -- quadrat == n

                        when 72 => mem_addr <= to_unsigned(72, 10);  mem_din <= x"4010"; mem_we <= '1';  -- BN
                        when 73 => mem_addr <= to_unsigned(73, 10);  mem_din <= x"005C"; mem_we <= '1';  -- n > quadrat

                        when 74 => mem_addr <= to_unsigned(74, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 75 => mem_addr <= to_unsigned(75, 10);  mem_din <= x"0052"; mem_we <= '1';  -- quadrat > n

-- quadrat == n

                        when 76 => mem_addr <= to_unsigned(76, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 77 => mem_addr <= to_unsigned(77, 10);  mem_din <= x"0009"; mem_we <= '1';  -- middle

                        when 78 => mem_addr <= to_unsigned(78, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 79 => mem_addr <= to_unsigned(79, 10);  mem_din <= x"0006"; mem_we <= '1';  -- sqrt

                        when 80 => mem_addr <= to_unsigned(80, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 81 => mem_addr <= to_unsigned(81, 10);  mem_din <= x"0066"; mem_we <= '1';  -- Finished

-- quadrat > n

                        when 82 => mem_addr <= to_unsigned(82, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 83 => mem_addr <= to_unsigned(83, 10);  mem_din <= x"0009"; mem_we <= '1';  -- middle

                        when 84 => mem_addr <= to_unsigned(84, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ
                        when 85 => mem_addr <= to_unsigned(85, 10);  mem_din <= x"0066"; mem_we <= '1';  -- Finished

                        when 86 => mem_addr <= to_unsigned(86, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        when 87 => mem_addr <= to_unsigned(87, 10);  mem_din <= x"FFFF"; mem_we <= '1';  -- -1

                        when 88 => mem_addr <= to_unsigned(88, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 89 => mem_addr <= to_unsigned(89, 10);  mem_din <= x"0005"; mem_we <= '1';  -- right

                        when 90 => mem_addr <= to_unsigned(90, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 91 => mem_addr <= to_unsigned(91, 10);  mem_din <= x"000A"; mem_we <= '1';  -- start of loop

-- quadrat < n

                        when 92 => mem_addr <= to_unsigned(92, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 93 => mem_addr <= to_unsigned(93, 10);  mem_din <= x"0009"; mem_we <= '1';  -- middle

                        when 94 => mem_addr <= to_unsigned(94, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 95 => mem_addr <= to_unsigned(95, 10);  mem_din <= x"0006"; mem_we <= '1';  -- sqrt

                        when 96 => mem_addr <= to_unsigned(96, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        when 97 => mem_addr <= to_unsigned(97, 10);  mem_din <= x"0001"; mem_we <= '1';

                        when 98 => mem_addr <= to_unsigned(98, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 99 => mem_addr <= to_unsigned(99, 10);  mem_din <= x"0004"; mem_we <= '1';  -- left

                        when 100 => mem_addr <= to_unsigned(100, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 101 => mem_addr <= to_unsigned(101, 10);  mem_din <= x"000A"; mem_we <= '1';  -- start of loop

                        when 102 => mem_addr <= to_unsigned(102, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 103 => mem_addr <= to_unsigned(103, 10);  mem_din <= x"0006"; mem_we <= '1';  -- sqrt
                            
                        when 104 => mem_addr <= to_unsigned(104, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 105 => mem_addr <= to_unsigned(105, 10);  mem_din <= x"02C2"; mem_we <= '1';  -- back to main

-- HALT


-- *****************************************
-- MUL: Parameter
-- *****************************************

                        when 200 => mem_addr <= to_unsigned(200, 10);  mem_din <= x"0017"; mem_we <= '1';  -- a
                        when 201 => mem_addr <= to_unsigned(201, 10);  mem_din <= x"000E"; mem_we <= '1';  -- b
                        when 202 => mem_addr <= to_unsigned(202, 10);  mem_din <= x"0000"; mem_we <= '1';  -- result

-- *****************************************
-- MUL
-- *****************************************

                        when 203 => mem_addr <= to_unsigned(203, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 204 => mem_addr <= to_unsigned(204, 10);  mem_din <= x"00C9"; mem_we <= '1';  -- b

                        when 205 => mem_addr <= to_unsigned(205, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ
                        when 206 => mem_addr <= to_unsigned(206, 10);  mem_din <= x"00E9"; mem_we <= '1';  -- return result

                        when 207 => mem_addr <= to_unsigned(207, 10);  mem_din <= x"4020"; mem_we <= '1';  -- BE
                        when 208 => mem_addr <= to_unsigned(208, 10);  mem_din <= x"00DB"; mem_we <= '1';  -- b is even, b & 1 == 0

                        when 209 => mem_addr <= to_unsigned(209, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 210 => mem_addr <= to_unsigned(210, 10);  mem_din <= x"00C8"; mem_we <= '1';  -- a

                        when 211 => mem_addr <= to_unsigned(211, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 212 => mem_addr <= to_unsigned(212, 10);  mem_din <= x"00D8"; mem_we <= '1';  -- copy a to add

                        when 213 => mem_addr <= to_unsigned(213, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 214 => mem_addr <= to_unsigned(214, 10);  mem_din <= x"00CA"; mem_we <= '1';  -- result

                        when 215 => mem_addr <= to_unsigned(215, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        when 216 => mem_addr <= to_unsigned(216, 10);  mem_din <= x"0000"; mem_we <= '1';  -- a

                        when 217 => mem_addr <= to_unsigned(217, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 218 => mem_addr <= to_unsigned(218, 10);  mem_din <= x"00CA"; mem_we <= '1';  -- result

                        when 219 => mem_addr <= to_unsigned(219, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 220 => mem_addr <= to_unsigned(220, 10);  mem_din <= x"00C8"; mem_we <= '1';  -- a

                        when 221 => mem_addr <= to_unsigned(221, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 222 => mem_addr <= to_unsigned(222, 10);  mem_din <= x"0000"; mem_we <= '1';

                        when 223 => mem_addr <= to_unsigned(223, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 224 => mem_addr <= to_unsigned(224, 10);  mem_din <= x"00C8"; mem_we <= '1';  -- result

                        when 225 => mem_addr <= to_unsigned(225, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 226 => mem_addr <= to_unsigned(226, 10);  mem_din <= x"00C9"; mem_we <= '1';  -- a

                        when 227 => mem_addr <= to_unsigned(227, 10);  mem_din <= x"3580"; mem_we <= '1';  -- SHL
                        when 228 => mem_addr <= to_unsigned(228, 10);  mem_din <= x"0000"; mem_we <= '1';

                        when 229 => mem_addr <= to_unsigned(229, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 230 => mem_addr <= to_unsigned(230, 10);  mem_din <= x"00C9"; mem_we <= '1';  -- result

                        when 231 => mem_addr <= to_unsigned(231, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 232 => mem_addr <= to_unsigned(232, 10);  mem_din <= x"00CB"; mem_we <= '1';  -- Loop

                        when 233 => mem_addr <= to_unsigned(233, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 234 => mem_addr <= to_unsigned(234, 10);  mem_din <= x"00CA"; mem_we <= '1';  -- result

                        when 235 => mem_addr <= to_unsigned(235, 10);  mem_din <= x"6000"; mem_we <= '1';  -- RETURN

                        when 240 => mem_addr <= to_unsigned(240, 10);  mem_din <= x"0000"; mem_we <= '1';  -- Calculate SQRT of this number
                            
-- end of SQRT ----------------------------------------
-- *************************************************************************************************************
                            


                            
-- Test routine...

                        when 600 => mem_addr <= to_unsigned(600, 10);  mem_din <= x"1400"; mem_we <= '1';  -- LDA
                        when 601 => mem_addr <= to_unsigned(601, 10);  mem_din <= x"0000"; mem_we <= '1';                            

                        when 602 => mem_addr <= to_unsigned(602, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 603 => mem_addr <= to_unsigned(603, 10);  mem_din <= x"03B6"; mem_we <= '1';  
                            
                        -- when 602 => mem_addr <= to_unsigned(602, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        -- when 603 => mem_addr <= to_unsigned(603, 10);  mem_din <= x"15A0"; mem_we <= '1';  -- -60000
                            
                        -- when 604 => mem_addr <= to_unsigned(604, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ
                        -- when 605 => mem_addr <= to_unsigned(605, 10);  mem_din <= x"0262"; mem_we <= '1';  -- load again
                            
                        when 604 => mem_addr <= to_unsigned(604, 10);  mem_din <= x"F000"; mem_we <= '1';  -- NOP
                            
                        when 605 => mem_addr <= to_unsigned(605, 10);  mem_din <= x"F000"; mem_we <= '1';  -- NOP
                            
-- main ------------------------------------------------
                            
                        when 700 => mem_addr <= to_unsigned(700, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 701 => mem_addr <= to_unsigned(701, 10);  mem_din <= x"0320"; mem_we <= '1';  -- 800

                        when 702 => mem_addr <= to_unsigned(702, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 703 => mem_addr <= to_unsigned(703, 10);  mem_din <= x"00F0"; mem_we <= '1';  -- calculate SQRT of this number
                            
                        -- when 702 => mem_addr <= to_unsigned(702, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        -- when 703 => mem_addr <= to_unsigned(703, 10);  mem_din <= x"0258"; mem_we <= '1'; -- 600 Test-Routine

                        when 704 => mem_addr <= to_unsigned(704, 10);  mem_din <= x"5000"; mem_we <= '1';  -- CALL SQRT
                        when 705 => mem_addr <= to_unsigned(705, 10);  mem_din <= x"000A"; mem_we <= '1';  -- 10
                            
                        when 706 => mem_addr <= to_unsigned(706, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 707 => mem_addr <= to_unsigned(707, 10);  mem_din <= x"0002"; mem_we <= '1';  -- set output register

-- com ---------------------------------------------------

                        when 800 => mem_addr <= to_unsigned(800, 10);  mem_din <= x"1400"; mem_we <= '1';  -- LDA
                        when 801 => mem_addr <= to_unsigned(801, 10);  mem_din <= x"0000"; mem_we <= '1';

                        when 802 => mem_addr <= to_unsigned(802, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 803 => mem_addr <= to_unsigned(803, 10);  mem_din <= x"0002"; mem_we <= '1';  -- set output register to 0

-- wait for first part: loop until valid bit == 1

                        when 804 => mem_addr <= to_unsigned(804, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 805 => mem_addr <= to_unsigned(805, 10);  mem_din <= x"0003"; mem_we <= '1';  -- load input register

                        when 806 => mem_addr <= to_unsigned(806, 10);  mem_din <= x"3480"; mem_we <= '1';  -- AND
                        when 807 => mem_addr <= to_unsigned(807, 10);  mem_din <= x"0100"; mem_we <= '1';  -- check valid bit

                        when 808 => mem_addr <= to_unsigned(808, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ
                        when 809 => mem_addr <= to_unsigned(809, 10);  mem_din <= x"0324"; mem_we <= '1';  -- load again

-- receive first part: wait  and load first part of parameter

                        when 810 => mem_addr <= to_unsigned(810, 10);  mem_din <= x"5000"; mem_we <= '1';  -- CALL to test subroutine
                        when 811 => mem_addr <= to_unsigned(811, 10);  mem_din <= x"0384"; mem_we <= '1';

                        when 812 => mem_addr <= to_unsigned(812, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 813 => mem_addr <= to_unsigned(813, 10);  mem_din <= x"0003"; mem_we <= '1';  -- load input register

                        when 814 => mem_addr <= to_unsigned(814, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 815 => mem_addr <= to_unsigned(815, 10);  mem_din <= x"03B6"; mem_we <= '1';  -- store first part in 3B6

-- wait for second part ---- wait for 0 first

                        when 816 => mem_addr <= to_unsigned(816, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 817 => mem_addr <= to_unsigned(817, 10);  mem_din <= x"0003"; mem_we <= '1';  -- load input register
                            
                        when 818 => mem_addr <= to_unsigned(818, 10);  mem_din <= x"3480"; mem_we <= '1';  -- AND
                        when 819 => mem_addr <= to_unsigned(819, 10);  mem_din <= x"0100"; mem_we <= '1';  -- check valid bit

                        when 820 => mem_addr <= to_unsigned(820, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ
                        when 821 => mem_addr <= to_unsigned(821, 10);  mem_din <= x"0338"; mem_we <= '1';
                            
                        when 822 => mem_addr <= to_unsigned(822, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 823 => mem_addr <= to_unsigned(823, 10);  mem_din <= x"0330"; mem_we <= '1';
                            
-- now check valid bit
                            
                        when 824 => mem_addr <= to_unsigned(824, 10);  mem_din <= x"5000"; mem_we <= '1';  -- CALL to test subroutine
                        when 825 => mem_addr <= to_unsigned(825, 10);  mem_din <= x"0384"; mem_we <= '1';
                            
                        when 826 => mem_addr <= to_unsigned(826, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 827 => mem_addr <= to_unsigned(827, 10);  mem_din <= x"0003"; mem_we <= '1';  -- load input register
                            

                        when 828 => mem_addr <= to_unsigned(828, 10);  mem_din <= x"3480"; mem_we <= '1';  -- AND
                        when 829 => mem_addr <= to_unsigned(829, 10);  mem_din <= x"0100"; mem_we <= '1';  -- check valid bit

                        when 830 => mem_addr <= to_unsigned(830, 10);  mem_din <= x"4018"; mem_we <= '1';  -- BZ
                        when 831 => mem_addr <= to_unsigned(831, 10);  mem_din <= x"0338"; mem_we <= '1';

                        -- when 830 => mem_addr <= to_unsigned(822, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA          // Loop back
                        -- when 831 => mem_addr <= to_unsigned(823, 10);  mem_din <= x"0338"; mem_we <= '1';
                            
                        when 832 => mem_addr <= to_unsigned(832, 10);  mem_din <= x"F000"; mem_we <= '1';  -- NOP
                        when 833 => mem_addr <= to_unsigned(833, 10);  mem_din <= x"F000"; mem_we <= '1';  -- NOP

-- receive second part

                        when 834 => mem_addr <= to_unsigned(834, 10);  mem_din <= x"5000"; mem_we <= '1';  -- CALL to wait subroutine
                        when 835 => mem_addr <= to_unsigned(835, 10);  mem_din <= x"0384"; mem_we <= '1';

                        when 836 => mem_addr <= to_unsigned(836, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 837 => mem_addr <= to_unsigned(837, 10);  mem_din <= x"0003"; mem_we <= '1';  -- load input register

                        when 838 => mem_addr <= to_unsigned(838, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 839 => mem_addr <= to_unsigned(839, 10);  mem_din <= x"03B7"; mem_we <= '1';  -- store socond part in 03B7
                            
                        -- when 840 => mem_addr <= to_unsigned(840, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        -- when 841 => mem_addr <= to_unsigned(841, 10);  mem_din <= x"0258"; mem_we <= '1';
                            
                        when 840 => mem_addr <= to_unsigned(840, 10);  mem_din <= x"F000"; mem_we <= '1';  -- NOP
                        when 841 => mem_addr <= to_unsigned(841, 10);  mem_din <= x"F000"; mem_we <= '1';  -- NOP
                            
                        when 842 => mem_addr <= to_unsigned(842, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 843 => mem_addr <= to_unsigned(843, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010

                        when 844 => mem_addr <= to_unsigned(844, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 845 => mem_addr <= to_unsigned(845, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010

                        when 846 => mem_addr <= to_unsigned(846, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 847 => mem_addr <= to_unsigned(847, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010

                        when 848 => mem_addr <= to_unsigned(848, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 849 => mem_addr <= to_unsigned(849, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010

                        when 850 => mem_addr <= to_unsigned(850, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 851 => mem_addr <= to_unsigned(851, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010

                        when 852 => mem_addr <= to_unsigned(852, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 853 => mem_addr <= to_unsigned(853, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010
                            
                        when 854 => mem_addr <= to_unsigned(854, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 855 => mem_addr <= to_unsigned(855, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010

                        when 856 => mem_addr <= to_unsigned(856, 10);  mem_din <= x"35C0"; mem_we <= '1';  -- SHL
                        when 857 => mem_addr <= to_unsigned(857, 10);  mem_din <= x"0000"; mem_we <= '1';  -- 1010 1010 1010 1010

                        when 858 => mem_addr <= to_unsigned(858, 10);  mem_din <= x"2800"; mem_we <= '1';  -- STAM
                        when 859 => mem_addr <= to_unsigned(859, 10);  mem_din <= x"0361"; mem_we <= '1';  -- to the add

-- part number 1

                        when 860 => mem_addr <= to_unsigned(860, 10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 861 => mem_addr <= to_unsigned(861, 10);  mem_din <= x"03B6"; mem_we <= '1';  -- first part

                        when 862 => mem_addr <= to_unsigned(862, 10);  mem_din <= x"3480"; mem_we <= '1';  -- AND
                        when 863 => mem_addr <= to_unsigned(863, 10);  mem_din <= x"00FF"; mem_we <= '1';  -- only data bits

                        when 864 => mem_addr <= to_unsigned(864, 10);  mem_din <= x"3440"; mem_we <= '1';  -- ADD
                        when 865 => mem_addr <= to_unsigned(865, 10);  mem_din <= x"0000"; mem_we <= '1';

                        when 866 => mem_addr <= to_unsigned(866, 10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 867 => mem_addr <= to_unsigned(867, 10);  mem_din <= x"02BE"; mem_we <= '1';  -- back to main
                            
 -- end of com
                            
                       ------------------------- Wait Loop -----------------------------------------
                            
                        when 900 => mem_addr <= to_unsigned(900,10); mem_din <= x"2800"; mem_we <= '1';  -- STA
                        when 901 => mem_addr <= to_unsigned(901,10); mem_din <= x"03B5"; mem_we <= '1';  -- 950
                            
                        when 902  => mem_addr <= to_unsigned(902,10); mem_din <= x"1400"; mem_we <= '1';  -- LDA
                        when 903  => mem_addr <= to_unsigned(903,10);  mem_din <= x"0350"; mem_we <= '1'; -- x200
                            
                        when 904 => mem_addr <= to_unsigned(904,10); mem_din <= x"2800"; mem_we <= '1';  -- STA
                        when 905 => mem_addr <= to_unsigned(905,10); mem_din <= x"03B7"; mem_we <= '1';  --
                            
                        when 906  => mem_addr <= to_unsigned(906,10); mem_din <= x"1800"; mem_we <= '1';  -- LDA
                        when 907  => mem_addr <= to_unsigned(907,10);  mem_din <= x"03B7"; mem_we <= '1'; -- #1
                            
                        when 908  => mem_addr <= to_unsigned(908,10); mem_din <= x"3440"; mem_we <= '1';   -- ADD
                        when 909  => mem_addr <= to_unsigned(909,10);  mem_din <= x"FFFF"; mem_we <= '1'; -- #1
                            
                        when 910 => mem_addr <= to_unsigned(910,10); mem_din <= x"2800"; mem_we <= '1';  -- STA
                        when 911 => mem_addr <= to_unsigned(911,10); mem_din <= x"03B7"; mem_we <= '1';  -- 2
                            
                        when 912  => mem_addr <= to_unsigned(912,10); mem_din <= x"4018"; mem_we <= '1';   -- BZ
                        when 913  => mem_addr <= to_unsigned(913,10);  mem_din <= x"0394"; mem_we <= '1';  -- 916
                            
                        when 914  => mem_addr <= to_unsigned(914,10);  mem_din <= x"4000"; mem_we <= '1';  -- BA
                        when 915  => mem_addr <= to_unsigned(915,10);  mem_din <= x"038A"; mem_we <= '1';  -- 906
                            
                        when 916  => mem_addr <= to_unsigned(916,10);  mem_din <= x"1800"; mem_we <= '1';  -- LDAM
                        when 917  => mem_addr <= to_unsigned(917,10);  mem_din <= x"03B5"; mem_we <= '1'; -- #1
                            
                        when 918  => mem_addr <= to_unsigned(918,10); mem_din <= x"6000"; mem_we <= '1';   -- RET
                            
                            
    -- Fertig -> Start ausführen
                        when 919 => state <= FETCH;

                        when others => null;
                    end case;

-- Zähler weiterschalten
                    if init_counter < 919 then
                        init_counter <= init_counter + 1;
                    end if;

                ------------------------------------------------------------
                when FETCH =>
                    mem_addr <= pc;
                    state    <= READ_WAIT_FETCH;
                    if (pc(9 downto 0) = DEBUG_ADDR) then
                        -- debug_reg <= (others => '1');                -- no operand for NOP or RET
                        debug_reg(0) <= '1';
                    end if;

                when READ_WAIT_FETCH =>
                    state    <= DECODE;
                    if (stop = '0') then
                        pc       <= pc + 1;
                    end if;

                when DECODE =>
                    ir       <= mem_dout;
                    r := number_operandReads(mem_dout);
                    reads_left <= r;
                    if (r = 0) then
                        state <= EXECUTE;                -- no operand for NOP or RET
                    else
                        state <= READ_WAIT_FIRST_OPERAND;         -- first level operand-Fetch
                        mem_addr <= pc;
                    end if;
                    
                when READ_WAIT_FIRST_OPERAND =>
                    state <= READ_FIRST_OPERAND;

                when READ_FIRST_OPERAND =>
                    if (reads_left = 1) then
                        operand <= mem_dout;
                        state <= EXECUTE;                -- we are finished with operand loading for LDA, CALL, Branch or MOP
                    else
                        state <= READ_WAIT_SECOND_OPERAND;         -- second level operand-fetch
                        mem_addr <= unsigned(mem_dout(9 downto 0));
                    end if;
                    
                when READ_WAIT_SECOND_OPERAND =>
                    state <= READ_SECOND_OPERAND;
                    
                when READ_SECOND_OPERAND =>
                    if (reads_left = 2) then
                        operand <= mem_dout;
                        state <= EXECUTE;                -- we are finished with operand loading for LDAM, STAM
                    else
                        state <= READ_WAIT_THIRD_OPERAND;         -- third level operand-fetch
                        mem_addr <= unsigned(mem_dout(9 downto 0));
                    end if;
                    
                when READ_WAIT_THIRD_OPERAND =>         -- now we are finished with operand loading for LDAMM and STAMM
                    state <= READ_THIRD_OPERAND;
                    
                when READ_THIRD_OPERAND =>
                    operand <= mem_dout;
                    state <= EXECUTE;
                    

                when EXECUTE =>
                    -- debug_reg <= ir;
                    if ir(15 downto 12) = "0100" then           -- Branch
                        case ir(5 downto 3) is
                            when "000" =>                       -- BA
                                pc <= unsigned(operand(9 downto 0));
                                
                            when "011" =>                       -- BZ
                                if accu = x"0000" then
                                    pc <= unsigned(operand(9 downto 0));
                                else
                                    pc <= pc +1;
                                end if;
                                
                            when "001" =>                       -- BC
                                if carryFlag = '1' then
                                    pc <= unsigned(operand(9 downto 0));
                                else
                                    pc <= pc +1;
                                end if;
                                
                            when "010" =>                       -- BN
                                if accu(15) = '1' then
                                    pc <= unsigned(operand(9 downto 0));
                                else
                                    pc <= pc +1;
                                end if;
                                
                            when "100" =>                       -- BE
                                if accu (0) = '0' then
                                    pc <= unsigned(operand(9 downto 0));
                                else
                                    pc <= pc +1;
                                end if;
                                
                            when others =>
                                pc <= pc + 1;
                                
                        end case;
                    else
                        case ir(15 downto 12) is
                            when "1111" =>                          -- NOP
                                --- do nothing
                                
                            when "0001" =>                          -- LDA, each variant of LDA...because we loaded the correct operand to store in the accu
                                next_accu(15 downto 0) := unsigned(operand);
                                next_accu(16) := '0';
                                pc       <= pc + 1;
                                writeAccu := true;
                                --led_reg <= operand(7 downto 0);
                                
                            when "0011" =>                          -- MOP
                                writeAccu := true;
                                case ir(9 downto 6) is
                                    when "0001" =>                  -- ADD
                                        next_accu := ('0' & unsigned(accu)) + ('0' & unsigned(operand));
                                        pc       <= pc + 1;
                                        --led_reg <= std_logic_vector(unsigned(accu) + unsigned(operand))(7 downto 0);
                                        
                                    when "0010" =>                   -- AND
                                        next_accu(15 downto 0) := (unsigned(accu) and unsigned(operand));
                                        next_accu(16) := '0';
                                        pc       <= pc + 1;
                                        --led_reg <= std_logic_vector(unsigned(accu) and unsigned(operand))(7 downto 0);
                                        
                                    when "0011" =>                   -- OR
                                        next_accu(15 downto 0) := (unsigned(accu) or unsigned(operand));
                                        next_accu(16) := '0';
                                        pc       <= pc + 1;
                                        --led_reg <= std_logic_vector(unsigned(accu) or unsigned(operand))(7 downto 0);
                                        
                                    when "0101" =>                   -- NOT
                                        next_accu(15 downto 0) := not(unsigned(accu));
                                        next_accu(16) := '0';
                                        pc       <= pc + 1;
                                        --led_reg <= std_logic_vector(not unsigned(accu))(7 downto 0);
                                        
                                    when "0110" =>
                                        next_accu(15 downto 0) := (unsigned(accu) srl 1);
                                        next_accu(16) := '0';
                                        pc       <= pc + 1;
                                        --led_reg <= std_logic_vector(unsigned(accu) srl 1)(7 downto 0);
                                        
                                    when "0111" =>
                                        next_accu(15 downto 0) := (unsigned(accu) sll 1);
                                        next_accu(16) := '0';
                                        pc       <= pc + 1;
                                        --led_reg <= std_logic_vector(unsigned(accu) sll 1)(7 downto 0);
                                        
                                    when others =>
                                        pc       <= pc + 1;

                                end case;
                            when "0010" =>                                          -- STAM
                                mem_addr <= unsigned(operand(9 downto 0));
                                mem_din  <= accu;
                                mem_we   <= '1';
                                pc       <= pc + 1;
                                if (unsigned(operand(9 downto 0))) = 2 then
                                    regOutputRegister <= accu(7 downto 0);
                                end if;
                            when "0101" =>                                      -- CALL
                                stack <= pc+1;
                                pc <= unsigned(operand(9 downto 0));
                            when "0110" =>                                  -- RET
                                pc <= stack;
                            when others =>
                                stop <= '1';
                                -- debug_reg <= (others => '0');
                        end case;
                    end if;
                    if (writeAccu = true) then
                        accu <= (std_logic_vector(next_accu(15 downto 0)));
                        carryFlag <= next_accu(16);
                    end if;
                    led_reg <= accu(7 downto 0);
                    state <= FETCH;
            end case;
        end if;
    end process;
    outputRegister <= regOutputRegister;
end Behavioral;


